<?php

namespace App\Http\Controllers;

use App\Models\RestaurantDish;
use Illuminate\Http\Request;

class RestaurantDishController extends Controller
{
    public function index(Request $request)
    {
        // Si es AJAX → devolver JSON (para loadDishes)
        if ($request->ajax()) {
            $search = $request->get('search', '');
            
            $dishes = RestaurantDish::when($search, function ($query, $search) {
                    $query->where('title', 'like', "%$search%");
                })
                ->orderByDesc('id')
                ->paginate(10);

            return response()->json([
                'data' => $dishes->items(),
                'total' => $dishes->total(),
                'current_page' => $dishes->currentPage(),
                'last_page' => $dishes->lastPage(),
            ]);
        }

        $dishes = RestaurantDish::orderBy('id', 'desc')->get();
        return view('admin.dishes.index', compact('dishes'));
    }

    public function create()
    {
        return view('admin.restaurants.dishes.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        $dish = new RestaurantDish($request->only(['title', 'price']));

        if ($request->hasFile('image')) {
            $imageName = time() . '.' . $request->image->extension();
            $request->image->move(public_path('img/restaurant-dishes'), $imageName);
            $dish->image = $imageName;
        }

        $dish->save();

        return response()->json(['message' => 'Plato registrado correctamente.']);
    }

    /**
     * Mostrar una actividad específica
     */
    public function show($id)
    {
        $restaurantDish = RestaurantDish::findOrFail($id);
        return response()->json($restaurantDish);
    }

    public function update(Request $request, RestaurantDish $dish)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        $dish->fill($request->only(['title', 'price']));

        if ($request->hasFile('image')) {
            $imageName = time() . '.' . $request->image->extension();
            $request->image->move(public_path('img/restaurant-dishes'), $imageName);
            $dish->image = $imageName;
        }

        $dish->save();

        return response()->json(['message' => 'Plato actualizado correctamente.']);
    }

    public function destroy($dishId)
    {
        $restaurantDish = RestaurantDish::find($dishId);

        if ($restaurantDish->image && file_exists(public_path('img/restaurant-dishes/' . $restaurantDish->image))) {
            unlink(public_path('img/restaurant-dishes/' . $restaurantDish->image));
        }

        $restaurantDish->delete();

        return response()->json(['message' => 'Plato eliminado correctamente.']);
    }
}
