<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Activity;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class ActivityController extends Controller
{
    /**
     * Mostrar listado o datos vía AJAX
     */
    public function index(Request $request)
    {
        // Si es AJAX → devolver JSON (para loadActivities)
        if ($request->ajax()) {
            $search = $request->get('search', '');
            $activities = Activity::when($search, function ($query, $search) {
                    $query->where('title', 'like', "%$search%");
                })
                ->orderByDesc('id')
                ->paginate(10);

            return response()->json([
                'data' => $activities->items(),
                'total' => $activities->total(),
                'current_page' => $activities->currentPage(),
                'last_page' => $activities->lastPage(),
            ]);
        }

        // Si no es AJAX → retornar vista
        return view('admin.activities.index');
    }

    /**
     * Mostrar una actividad específica
     */
    public function show($id)
    {
        $activity = Activity::findOrFail($id);
        return response()->json($activity);
    }

    /**
     * Guardar una nueva actividad
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'duration' => 'required|string|max:255',
            'price' => 'nullable|numeric|min:0',
            'people_count' => 'nullable|integer|min:0',
            'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $activity = new Activity();
        $activity->title = $request->title;
        $activity->description = $request->description;
        $activity->duration = $request->duration;
        $activity->season = $request->season;
        $activity->difficulty_level = $request->difficulty_level;
        $activity->location = $request->location;
        $activity->itinerary = $request->itinerary;
        $activity->includes = $request->includes;
        $activity->not_includes = $request->not_includes;
        $activity->price = $request->price ?? 0;
        $activity->people_count = $request->people_count ?? 0;

        // Imagen
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = time() . '_' . Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('img/activities'), $filename);
            $activity->image = $filename;
        } else {
            $activity->image = 'default.jpg';
        }

        $activity->save();

        return response()->json(['message' => 'Actividad registrada correctamente.']);
    }

    /**
     * Actualizar actividad existente
     */
    public function update(Request $request, $id)
    {
        $activity = Activity::findOrFail($id);

        $request->validate([
            'title' => 'required|string|max:255',
            'duration' => 'required|string|max:255',
            'price' => 'nullable|numeric|min:0',
            'people_count' => 'nullable|integer|min:0',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp,gif|max:4096',
        ]);

        $activity->title = $request->title;
        $activity->description = $request->description;
        $activity->duration = $request->duration;
        $activity->season = $request->season;
        $activity->difficulty_level = $request->difficulty_level;
        $activity->location = $request->location;
        $activity->itinerary = $request->itinerary;
        $activity->includes = $request->includes;
        $activity->not_includes = $request->not_includes;
        $activity->price = $request->price ?? 0;
        $activity->people_count = $request->people_count ?? 0;

        // Imagen (reemplazo si hay nueva)
        if ($request->hasFile('image')) {
            // eliminar imagen anterior si no es default
            if ($activity->image && $activity->image !== 'default.jpg' && File::exists(public_path('img/activities/' . $activity->image))) {
                File::delete(public_path('img/activities/' . $activity->image));
            }

            $file = $request->file('image');
            $filename = time() . '_' . Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('img/activities'), $filename);
            $activity->image = $filename;
        }

        $activity->save();

        return response()->json(['message' => 'Actividad actualizada correctamente.']);
    }

    /**
     * Eliminar actividad
     */
    public function destroy($id)
    {
        $activity = Activity::findOrFail($id);

        // eliminar imagen
        if ($activity->image && $activity->image !== 'default.jpg' && File::exists(public_path('img/activities/' . $activity->image))) {
            File::delete(public_path('img/activities/' . $activity->image));
        }

        $activity->delete();

        return response()->json(['message' => 'Actividad eliminada correctamente.']);
    }

    /**
     * Exportar a PDF (opcional)
     */
    public function exportPdf(Request $request)
    {
        $activities = Activity::all();
        $html = view('admin.activities.pdf', compact('activities'))->render();

        $pdf = app('dompdf.wrapper');
        $pdf->loadHTML($html);
        return $pdf->download('actividades.pdf');
    }

    /**
     * Exportar a Excel (opcional)
     */
    public function exportExcel(Request $request)
    {
        $activities = Activity::all();
        $filename = 'actividades_' . date('Ymd_His') . '.csv';
        $handle = fopen($filename, 'w+');
        fputcsv($handle, ['ID', 'Título', 'Duración', 'Estación', 'Dificultad', 'Precio', 'Personas']);

        foreach ($activities as $a) {
            fputcsv($handle, [
                $a->id, $a->title, $a->duration, $a->season,
                $a->difficulty_level, $a->price, $a->people_count
            ]);
        }

        fclose($handle);
        return response()->download($filename)->deleteFileAfterSend(true);
    }

    /**
     * Mostrar listado en pagina home
     */
    public function indexPublic(Request $request)
    {
        $activities = Activity::orderByDesc('id')->get();

        return view('public.activities.index', compact('activities'));
    }

    /**
     * Mostrar listado en pagina home
     */
    public function listPublic(Request $request)
    {
        $activities = Activity::orderByDesc('id')
                ->get()
                ->take(3);

        return response()->json(['activities' => $activities]);
    }

    /**
     * Mostrar vista detalle de actividades
     */
    public function showPublic($id)
    {
        $activity = Activity::find($id);

        return view('public.activities.show', compact('activity'));
    }
}
